//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverTcp6Utils.cc 11949 $
// @(#) $Id: DwmMcroverTcp6Utils.cc 11949 2022-04-17 17:04:51Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverTcp6Utils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::Tcp6Utils class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <errno.h>
  #include <fcntl.h>
  #include <unistd.h>
}

#include <chrono>
#include <thread>

#include "DwmSvnTag.hh"
#include "DwmMcroverTargetHostConfig.hh"
#include "DwmMcroverTcp6Utils.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverTcp6Utils.cc 11949 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Tcp6Utils::GetAlerts(const AlertOrigin & origin,
                              const std::vector<TargetHostConfig> & hosts,
                              AlertBowl & alerts,
                              bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      map<Tcp6DstAddr,TcpTestConnectData>  dm, dmdeny;
      for (const auto & host : hosts) {
        if (! host.TCP6Ports().empty()) {
          for (auto port : host.TCP6Ports()) {
            dm[Tcp6DstAddr(host.Name(), host.Address6(), port)] =
              TcpTestConnectData();
          }
          for (auto port : host.TCP6PortsDenied()) {
            dmdeny[Tcp6DstAddr(host.Name(), host.Address6(), port)] =
              TcpTestConnectData();
          }
        }
      }
      if (! dm.empty()) {
        //  Test that we can connect to ports that should be listening.
        TestConnect(dm);
        for (const auto & dst : dm) {
          if (! dst.second.Passed()) {
            alerts.Add(origin, Tcp6ConnAlert(dst.first),
                       dst.second.LastFailed());
          }
        }
      }
      if (! dmdeny.empty()) {
        //  Test that we can't connect to ports that should not be listening.
        TestConnect(dmdeny);
        for (const auto & dst : dmdeny) {
          if (dst.second.Passed()) {
            alerts.Add(origin, Tcp6ConnDeniedAlert(dst.first),
                       dst.second.LastFailed());
          }
        }
      }
      return (! alerts.Empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void
    Tcp6Utils::TestConnect(map<Tcp6DstAddr,TcpTestConnectData> & addrs,
                           chrono::milliseconds timeout)
    {
      // for (auto & addr : addrs) { addr.second = false; }

      using Clock = chrono::system_clock;

      Clock::time_point  now = Clock::now();
      Clock::time_point  endTime = now + timeout;
      
      map<Tcp6DstAddr,int>  fds;

      //  open all sockets
      for (const auto & addr : addrs) {
        int  fd = socket(PF_INET6, SOCK_STREAM, 0);
        if (fd >= 0) {
          int  fdFlags = fcntl(fd, F_GETFL, 0);
          fdFlags |= O_NONBLOCK;
          if (fcntl(fd, F_SETFL, fdFlags) != -1) {
            fds[addr.first] = fd;
          }
          else {
            close(fd);
            fds[addr.first] = -1;
          }
        }
        else {
          fds[addr.first] = -1;
        }
      }

      size_t  divisor = 256;
      
      while (Clock::now() < endTime) {
        auto  sleepTime = timeout / divisor;
        if (sleepTime < chrono::milliseconds(10)) {
          sleepTime = chrono::milliseconds(10);
        }
        else if (sleepTime > chrono::milliseconds(200)) {
          sleepTime = chrono::milliseconds(200);
        }
        if (divisor > 2) {
          divisor >>= 1;
        }
        for (auto & addr : addrs) {
          if (! addr.second.Passed()) {
            //  We haven't connected yet.
            auto  statusIter = fds.find(addr.first);
            if (statusIter != fds.end()) {
              if (statusIter->second >= 0) {
                int connectrc = -1;
                if (statusIter->first.Address() != Ipv6Address(in6addr_any)) {
                  errno = 0;
                  connectrc =
                    connect(statusIter->second, statusIter->first.SockAddr(),
                            sizeof(sockaddr_in6));
                }
                else {
                  Ipv6Address  foundAddr =
                    Utils::GetHostIpv6Addr(statusIter->first.Name());
                  if (foundAddr != Ipv6Address(in6addr_any)) {
                    struct sockaddr_in6  sockAddr;
                    memcpy(&sockAddr, statusIter->first.SockAddr(),
                           sizeof(sockAddr));
                    sockAddr.sin6_addr = foundAddr;
                    sockAddr.sin6_family = AF_INET6;
#ifndef __linux__
                    sockAddr.sin6_len = sizeof(sockAddr);
#endif
                    errno = 0;
                    connectrc =
                      connect(statusIter->second, (struct sockaddr *)&sockAddr,
                              sizeof(sockAddr));
                  }
                }
                if (0 == connectrc) {
                  addr.second.LastPassed(Clock::to_time_t(now));
                  close(statusIter->second);
                  fds.erase(statusIter);
                }
                else if (EISCONN == errno) {
                  addr.second.LastPassed(Clock::to_time_t(now));
                  close(statusIter->second);
                  fds.erase(statusIter);
                }
              }
            }
          }
          if (fds.empty()) {
            break;
          }
        }
        if (fds.empty()) {
          break;
        }
        this_thread::sleep_for(sleepTime);
      }

      for (auto & fd : fds) {
        if (fd.second >= 0) {
          auto  dstit = addrs.find(fd.first);
          if (dstit != addrs.end()) {
            dstit->second.LastFailed(Clock::to_time_t(now));
          }
          close(fd.second);
        }
      }
      
      return;
    }
    
    
  }  // namespace Mcrover

}  // namespace Dwm

