//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverZFSUtils.cc 12022 $
// @(#) $Id: DwmMcroverZFSUtils.cc 12022 2022-05-30 19:03:23Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverZFSUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::ZFSUtils class implementation
//---------------------------------------------------------------------------

#include <cstdio> //  for popen(), fscanf(), fread()

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverZFSUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/src/DwmMcroverZFSUtils.cc 12022 $");

namespace Dwm {

  namespace Mcrover {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool PoolIsHealthy(const std::string & poolName,
                              std::string & status)
    {
      bool  rc = false;
      status = "UNKNOWN";
      std::string  cmd("zpool list -H -o health ");
      cmd += poolName;
      FILE  *p = popen(cmd.c_str(), "r");
      if (p) {
        char  buf[128] = { '\0' };
        if (fscanf(p, "%127s", buf) == 1) {
          status = buf;
          if (status == "ONLINE") {
            rc = true;
          }
          Syslog(LOG_INFO, "zpool '%s' is '%s'", poolName.c_str(), buf);
        }
        else {
          Syslog(LOG_ERR, "Failed to read zpool status from pipe");
        }
        pclose(p);
      }
      else {
        Syslog(LOG_ERR, "popen(\"%s\",\"r\") failed", cmd.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool PoolCapacity(const std::string & poolName,
                             int8_t & capacity)
    {
      bool  rc = false;
      capacity = -1;
      std::string  cmd("zpool list -H -o capacity ");
      cmd += poolName;
      FILE  *p = popen(cmd.c_str(), "r");
      if (p) {
        int8_t  cap = -1;
        if ((fscanf(p, "%hhu%%", &cap) == 1) && (cap >= 0)) {
          capacity = cap;
          rc = true;
          Syslog(LOG_INFO, "pool '%s' capacity is %hhu%%",
                 poolName.c_str(), capacity);
        }
        else {
          Syslog(LOG_ERR, "Failed to read zpool capacity for '%s'",
                 poolName.c_str());
        }
        pclose(p);
      }
      else {
        Syslog(LOG_ERR, "popen(\"%s\",\"r\") failed", cmd.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ZFSUtils::GetAlerts(const AlertOrigin & origin,
                             const LocalHostConfig & local,
                             AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      Ipv4Address  localAddr(Utils::ThisHostIpv4Addr());
      for (const auto zpool : local.Zpools()) {
        std::string  poolStatus;
        if (! PoolIsHealthy(zpool.first, poolStatus)) {
          alerts.Add(origin,
                     ZpoolAlert(localAddr, zpool.first, poolStatus),
                     time((time_t *)0));
          Syslog(LOG_ERR, "zpool health failed for %s", zpool.first.c_str());
        }
        else {
          Syslog(LOG_INFO, "zpool health passed for %s", zpool.first.c_str());
        }
        int8_t  poolCapacity;
        if (PoolCapacity(zpool.first, poolCapacity)) {
          if (poolCapacity >= zpool.second) {
            alerts.Add(origin,
                       ZpoolAlert(localAddr, zpool.first,
                                  (uint8_t)poolCapacity),
                       time((time_t *)0));
            Syslog(LOG_ERR, "zpool capacity failed for %s",
                   zpool.first.c_str());
          }
          else {
            Syslog(LOG_INFO, "zpool capacity passed for %s",
                   zpool.first.c_str());
          }
        }
      }
      return (! alerts.Empty());
    }
    
    
    
  }  // namespace Mcrover

}  // namespace Dwm
