//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.7/classes/tests/TestConfig.cc 12283 $
// @(#) $Id: TestConfig.cc 12283 2023-11-14 05:05:22Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestConfig.cc
//!  \author Daniel W. McRobb
//!  \brief Unit tests for Dwm::Mcrover::Config class
//---------------------------------------------------------------------------

#include <fstream>

#include "DwmUnitAssert.hh"
#include "DwmMcroverConfig.hh"

using namespace std;
using Dwm::Assertions;
using Dwm::Mcrover::TargetHostConfig;
using Dwm::Mcrover::PackConfig;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestHaveExpectedServers(const vector<TargetHostConfig> & servers)
{
  static const vector<string>  expected = {
    "localhost",
    "ria.rfdm.com",
    "kiva.rfdm.com",
    "www.rfdm.com",
    "pi1.rfdm.com",
    "waffles.rfdm.com",
    "us16xg.rfdm.com",
    "us16xg2.rfdm.com",
    "us24.rfdm.com",
    "us48poe.rfdm.com",
    "uapacpro1.rfdm.com",
    "airportexpress.rfdm.com",
    "ooma.rfdm.com",
    "appletv.rfdm.com",
    "www.google.com",
    "www.amazon.com",
    "icloud.com",
    "leoni.phacility.com",
    "weather.rfdm.com"
  };

  for (const auto & ex : expected) {
    auto  serv = find_if(servers.begin(), servers.end(),
                         [&] (const TargetHostConfig & thc)
                         { return (thc.Name() == ex); });
    if (UnitAssert(serv != servers.end())) {
      UnitAssert(! serv->Name().empty());
      if (serv->Name() == "www.rfdm.com") {
        if (UnitAssert(serv->WebAppTargets().size() == 3)) {
          UnitAssert(serv->WebAppTargets().front().URI()
                     == "http://www.rfdm.com/blog");
          UnitAssert(serv->WebAppTargets().front().AppName()
                     == "wordpress");
          UnitAssert(serv->WebAppTargets().back().URI()
                     == "http://www.spcarsplus.com/piwigo");
          UnitAssert(serv->WebAppTargets().back().AppName()
                     == "piwigo");
        }
      }
      if (serv->Name() == "weather.rfdm.com") {
        if (UnitAssert(! serv->CredencePeerTargets().empty())) {
          UnitAssert(serv->CredencePeerTargets().front().Port() == 2124);
          UnitAssert(serv->CredencePeerTargets().front().Name()
                     == "mcweatherd");
        }
      }
      if (serv->Name() == "kiva.rfdm.com") {
        if (UnitAssert(serv->WebAppTargets().size() == 1)) {
          UnitAssert(serv->WebAppTargets().front().AppName() == "plex");
          UnitAssert(! serv->WebAppTargets().front().Params().empty());
          UnitAssert(! serv->WebAppTargets().front().Xpath().empty());
        }
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestHaveDeniedPorts(const vector<TargetHostConfig> & servers)
{
  vector<string>  hosts = { "localhost", "ria.rfdm.com" };
  for (const auto & host : hosts) {
    auto  serv = find_if(servers.begin(), servers.end(),
                         [&] (const TargetHostConfig & thc)
                         { return (thc.Name() == host); });
    if (UnitAssert(serv != servers.end())) {
      if (UnitAssert(serv->TCP4PortsDenied().size() == 3)) {
        vector<uint16_t>  ports = { 21, 23, 79 };
        for (auto port : ports) {
          auto  piter = find_if(serv->TCP4PortsDenied().begin(),
                                serv->TCP4PortsDenied().end(),
                                [&] (const uint16_t p)
                                { return (p == port); });
          UnitAssert(piter != serv->TCP4PortsDenied().end());
        }
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestHavePackMembers(const PackConfig & packcfg)
{
  static const vector<pair<string,Dwm::Ipv4Address>>  members = {
    { "www",  Dwm::Ipv4Address("192.168.168.2") },
    { "ria",  Dwm::Ipv4Address("192.168.168.1") },
    { "kiva", Dwm::Ipv4Address("192.168.168.3") }
  };
      
  UnitAssert(packcfg.PackName() == "dwmpack");
  UnitAssert(packcfg.Members().size() == 3);
  for (const auto & m : members) {
    auto  mit = packcfg.Members().find(m.first);
    if (UnitAssert(mit != packcfg.Members().end())) {
      UnitAssert(mit->second.Name() == m.first);
      //  fixme UnitAssert(mit->second.Addr() == m.second);
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestHaveOtherPacks(const map<string,PackConfig> & otherpacks)
{
  UnitAssert(otherpacks.size() == 2);
  auto  it = otherpacks.find("jsm");
  UnitAssert(it != otherpacks.end());
  UnitAssert(it->second.Members().size() == 1);
  auto  mit = it->second.Members().find("jsmalpha");
  if (UnitAssert(mit != it->second.Members().end())) {
    UnitAssert(mit->second.Name() == "jsmalpha");
    if (UnitAssert(! mit->second.Addresses().empty())) {
      UnitAssert(mit->second.Addresses().begin()->address()
                 == boost::asio::ip::make_address("10.11.12.13"));
    }
  }

  it = otherpacks.find("mrm");
  UnitAssert(it != otherpacks.end());
  UnitAssert(it->second.Members().size() == 1);
  mit = it->second.Members().find("mrmalpha");
  if (UnitAssert(mit != it->second.Members().end())) {
    UnitAssert(mit->second.Name() == "mrmalpha");
    if (UnitAssert(! mit->second.Addresses().empty())) {
      UnitAssert(mit->second.Addresses().begin()->address()
                 == boost::asio::ip::make_address("10.5.5.5"));
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestHaveExpectedDumps(const map<string,uint64_t> & dumps)
{
  if (UnitAssert(dumps.size() == 3)) {
    auto  it = dumps.find("/dev/gpt/gprootfs");
    if (UnitAssert(it != dumps.end())) {
      UnitAssert(it->second == 108000);
    }
    it = dumps.find("/dev/gpt/gpusrfs");
    if (UnitAssert(it != dumps.end())) {
      UnitAssert(it->second == 108000);
    }
    it = dumps.find("/dev/gpt/gpvarfs");
    if (UnitAssert(it != dumps.end())) {
      UnitAssert(it->second == 108000);
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestJson(const Dwm::Mcrover::Config & config)
{
  using batcp = boost::asio::ip::tcp;

  nlohmann::json  j = config.ToJson();
  Dwm::Mcrover::Config  config2;
  if (UnitAssert(config2.FromJson(j))) {
    UnitAssert(config2.Service().Addresses().size() == 2);
    UnitAssert(config2.Service().Addresses().find(batcp::endpoint(batcp::v4(), 2123))
               != config2.Service().Addresses().end());
    UnitAssert(config2.Service().Addresses().find(batcp::endpoint(batcp::v6(), 2123))
               != config2.Service().Addresses().end());
    UnitAssert(config2.Servers().size() == 19);
    TestHaveExpectedServers(config2.Servers());
    TestHaveDeniedPorts(config2.Servers());
    TestHavePackMembers(config2.MyPack());
    TestHaveOtherPacks(config2.OtherPacks());
    TestHaveExpectedDumps(config2.Local().Dumps());
    UnitAssert(config2.Weather().Host() == "weather.rfdm.com");
    UnitAssert(config2.Weather().Port() == 21224);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int  rc = 1;

  string  cfgFile("./test.cfg");
  if (argc > 1) {
    cfgFile = argv[1];
  }

  // using Dwm::Mcrover::TCPAddress;

  using batcp = boost::asio::ip::tcp;

  Dwm::Mcrover::Config  config;
  if (UnitAssert(config.Parse(cfgFile))) {
    TestJson(config);
    
    UnitAssert(config.Service().Addresses().size() == 2);
    UnitAssert(config.Service().Addresses().find(batcp::endpoint(batcp::v4(), 2123))
               != config.Service().Addresses().end());
    UnitAssert(config.Service().Addresses().find(batcp::endpoint(batcp::v6(), 2123))
               != config.Service().Addresses().end());
    UnitAssert(config.Servers().size() == 19);
    TestHaveExpectedServers(config.Servers());
    TestHaveDeniedPorts(config.Servers());
    TestHavePackMembers(config.MyPack());
    TestHaveOtherPacks(config.OtherPacks());
    UnitAssert(config.Weather().Host() == "weather.rfdm.com");
    UnitAssert(config.Weather().Port() == 21224);
    
    ofstream  tmpofs("./tmp_config.cfg");
    if (UnitAssert(tmpofs)) {
      tmpofs << config;
      tmpofs.close();

      if (UnitAssert(config.Parse("./tmp_config.cfg"))) {
        UnitAssert(config.Service().Addresses().size() == 2);
        UnitAssert(config.Service().Addresses().find(batcp::endpoint(batcp::v4(), 2123))
                   != config.Service().Addresses().end());
        UnitAssert(config.Service().Addresses().find(batcp::endpoint(batcp::v6(), 2123))
                   != config.Service().Addresses().end());
        UnitAssert(config.Servers().size() == 19);
        TestHaveExpectedServers(config.Servers());
        TestHaveDeniedPorts(config.Servers());
        TestHavePackMembers(config.MyPack());
        TestHaveOtherPacks(config.OtherPacks());
        UnitAssert(config.Weather().Host() == "weather.rfdm.com");
        UnitAssert(config.Weather().Port() == 21224);
      }
      std::remove("./tmp_config.cfg");
    }
  }

  if (Assertions::Total().Failed()) {
    Assertions::Print(cerr, true);
  }
  else {
    cout << Assertions::Total() << " passed" << endl;
    rc = 0;
  }
  return rc;
}
