//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/include/DwmDnsLabel.hh 10084 $
// @(#) $Id: DwmDnsLabel.hh 10084 2018-01-26 04:22:01Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsLabel.hh
//!  \brief Dwm::Dns::Label class definition
//---------------------------------------------------------------------------

#ifndef _DWMDNSLABEL_HH_
#define _DWMDNSLABEL_HH_

#include <cstdint>
#include <string>

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  Encapsulates a DNS label.  See section 2.3.1 of RFC1035 for
    //!  details.
    //------------------------------------------------------------------------
    class Label
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor.  Creates an empty label.
      //----------------------------------------------------------------------
      Label() = default;

      //----------------------------------------------------------------------
      //!  Construct from a string.
      //----------------------------------------------------------------------
      Label(const std::string & label);
      
      //----------------------------------------------------------------------
      //!  Convert to a string.
      //----------------------------------------------------------------------
      operator std::string () const;
      
      //----------------------------------------------------------------------
      //!  Encodes the label into buffer @ pkt of length @c pktlen, starting
      //!  at @c ptr.  Returns the address immediately following the encoded
      //!  label in @c pkt on success.  Throws std::out_of_range if @c pkt
      //!  is too short to contain the encoded label.
      //----------------------------------------------------------------------
      uint8_t *Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen) const;
      
      //----------------------------------------------------------------------
      //!  Decodes the label from buffer @c pkt of length @c pktlen, starting
      //!  at @c ptr.  Returns the address immediately following the encoded
      //!  label in @c pkt on success.  Throws std::out_of_range if @c pkt
      //!  was too short to contain an encoded label.
      //----------------------------------------------------------------------
      const uint8_t *Decode(const uint8_t *pkt, const uint8_t *ptr,
                            uint16_t pktlen);

    protected:
      std::string  _value;
    };
    
  }  // namespace Dns

}  // namespace Dwm

#endif  // _DWMDNSLABEL_HH_
