%{
  //=========================================================================
  // @(#) $Name:$
  // @(#) $Id:$
  //===========================================================================
  //  Copyright (c) Daniel W. McRobb 2017
  //  All rights reserved.
  //
  //  Redistribution and use in source and binary forms, with or without
  //  modification, are permitted provided that the following conditions
  //  are met:
  //
  //  1. Redistributions of source code must retain the above copyright
  //     notice, this list of conditions and the following disclaimer.
  //  2. Redistributions in binary form must reproduce the above copyright
  //     notice, this list of conditions and the following disclaimer in the
  //     documentation and/or other materials provided with the distribution.
  //  3. The names of the authors and copyright holders may not be used to
  //     endorse or promote products derived from this software without
  //     specific prior written permission.
  //
  //  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
  //  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
  //  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
  //  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
  //  DAMAGE.
  //
  //  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
  //  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
  //  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
  //  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
  //  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  //  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
  //  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
  //  TRADEMARK OR OTHER RIGHTS.
  //=========================================================================
  
  //-------------------------------------------------------------------------
  //!  \file DwmCfg2JsonParse.yy
  //!  \brief Dwm::Cfg2Json class implementation and config file parser
  //-------------------------------------------------------------------------

  extern int cfg2jsonlex();

  #include <iostream>
  #include <iomanip>
  #include <mutex>
  #include <regex>
  #include <sstream>
  #include <utility>
  #include <nlohmann/json.hpp>

  #include "DwmCfg2Json.hh"
  
  extern "C" {
    extern int cfg2jsonerror(const char *arg, ...);
  }
  extern FILE *cfg2jsonin;

  static Dwm::Cfg2Json *g_cfg2json = 0;
%}

%define api.prefix {cfg2json}

%union {
  std::string                            *stringVal;
  std::pair<std::string,std::string>     *stringPairVal;
  std::pair<std::string,nlohmann::json>  *stringJsonPairVal;
  nlohmann::json                         *jsonVal;
}

%token<stringVal>         STRING

%type<jsonVal>            ConfigFile Stanzas StanzaContent
%type<stringPairVal>      NameValuePair
%type<stringJsonPairVal>  Stanza
%type<stringVal>          String

%%

ConfigFile: Stanzas
{
  g_cfg2json->JsonValue(*$1);
  delete $1;
};

Stanzas: Stanza
{
  $$ = new nlohmann::json();
  (*$$)[$1->first] = $1->second;
  delete $1;
}
| Stanzas Stanza
{
  (*$$)[$2->first] = $2->second;
  delete $2;
};

Stanza: String '{' StanzaContent '}'
{
  //  no slashes in stanza names
  if ($1->find_first_of('/') != std::string::npos) {
    std::cerr << "Illegal stanza name (contains slashes): \"" << *$1 << "\"\n";
    YYERROR;
  }
  $$ = new std::pair<std::string,nlohmann::json>(*$1,*$3);
  delete $1;
  delete $3;
}
| String '{' '}'
{
  //  no slashes in stanza names
  if ($1->find_first_of('/') != std::string::npos) {
    std::cerr << "Illegal stanza name (contains slashes): \"" << *$1 << "\"\n";
    YYERROR;
  }
  $$ = new std::pair<std::string,nlohmann::json>(*$1,nlohmann::json());
  delete $1;
};

StanzaContent: NameValuePair
{
  $$ = new nlohmann::json();
  (*$$)[$1->first] = $1->second;
  delete $1;
}
| Stanza
{
  $$ = new nlohmann::json();
  (*$$)[$1->first] = $1->second;
  delete $1;
}
| StanzaContent NameValuePair
{
  (*$$)[$2->first] = $2->second;
  delete $2;
}
| StanzaContent Stanza
{
  (*$$)[$2->first] = $2->second;
  delete $2;
};

NameValuePair: String ':' String ';'
{
  $$ = new std::pair<std::string,std::string>(*$1, *$3);
  delete $1;
  delete $3;
};

String: STRING
{
  $$ = $1;
}
| '"' STRING '"'
{
  $$ = $2;
}
| String '"' STRING '"'
{
  (*$$) += *$3;
  delete $3;
};

%%

using namespace std;

namespace Dwm {

  static mutex  g_cfg2jsonMutex;
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  Cfg2Json::Cfg2Json()
      : _jsonValue()
  {}

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool Cfg2Json::Parse(const std::string & confFilename)
  {
    bool  rc = false;
    lock_guard<mutex>  lock(g_cfg2jsonMutex);

    g_cfg2json = this;
    
    cfg2jsonin = fopen(confFilename.c_str(), "r");
    if (cfg2jsonin) {
      if (cfg2jsonparse() == 0) {
        rc = true;
      }
      fclose(cfg2jsonin);
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  const nlohmann::json & Cfg2Json::JsonValue() const
  {
    return _jsonValue;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  const nlohmann::json & Cfg2Json::JsonValue(const nlohmann::json & value)
  {
    _jsonValue = value;
    return _jsonValue;
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  const nlohmann::json *Cfg2Json::Find(const string & path) const
  {
    vector<string>  &&pathComps = GetPathComps(path);
    const nlohmann::json  *jvr = &_jsonValue;
    int  i = 0;
    for ( ; i < pathComps.size(); ++i) {
    	auto  it = jvr->find(pathComps[i]);
      if (it != jvr->end()) {
        jvr = &(*it);
      }
      else {
        jvr = 0;
        break;
      }
    }
    return jvr;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  const nlohmann::json *Cfg2Json::Find(const nlohmann::json *jvp,
                                       const string & path) const
  {
    const nlohmann::json *rc = nullptr;
    if (jvp) {
      vector<string>  &&pathComps = GetPathComps(path);
      rc = jvp;
      int  i;
      for ( ; i < pathComps.size(); ++i) {
        auto  it = rc->find(pathComps[i]);
        if (it != rc->end()) {
          rc = &(*it);
        }
        else {
          rc = nullptr;
          break;
        }
      }
    }
    return rc;
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void Cfg2Json::ToConfString(ostream & os, const nlohmann::json & jv,
                              int indent)
  {
    regex   r("[{};: \t]", regex::ECMAScript|regex::optimize);
    if (jv.is_string()) {
      os << ": ";
      smatch  sm;
      string  jvs(jv.get<string>());
      if (regex_search(jvs, sm, r) && (sm.size() > 0)) {
        os << "\"" << jvs << "\";\n";
      }
      else {
        os << jvs << ";\n";
      }
    }
    else {
      for (auto & m : jv.items()) {
        smatch  sm;
        if (regex_search(m.key(), sm, r) && (sm.size() > 0)) {
          os << setw(indent) << "" << "\"" << m.key() << "\"";
        }
        else {
          os << setw(indent) << "" << m.key();
        }
        if (! jv[m.key()].is_string()) {
          os << " {\n";
        }
        ToConfString(os, jv[m.key()], indent + 2);
        if (! jv[m.key()].is_string()) {
          os << setw(indent) << "" << "}\n";
          if (indent == 0) {
            os << '\n';
          }
        }
      }
#if 0
      auto  members = jv.getMemberNames();
      for (auto m : members) {
        smatch  sm;
        if (regex_search(m, sm, r) && (sm.size() > 0)) {
          os << setw(indent) << "" << "\"" << m << "\"";
        }
        else {
          os << setw(indent) << "" << m;
        }
        if (! jv[m].isString()) {
          os << " {\n";
        }
        // cout << "member " << m << '\n';
        ToConfString(os, jv[m], indent + 2);
        if (! jv[m].isString()) {
          os << setw(indent) << "" << "}\n";
          if (indent == 0) {
            os << '\n';
          }
        }
      }
#endif
    }
    return;
  }
      
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  std::string Cfg2Json::ToConfString() const
  {
    ostringstream  os;
    ToConfString(os, _jsonValue);
    return os.str();
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  vector<string> Cfg2Json::GetPathComps(const string & path)
  {
    vector<string>  rc;
    string          p(path);
    regex           r("[^/]+", regex::ECMAScript|regex::optimize);
    smatch          sm;
    while (regex_search(p, sm, r)) {
      if (sm.size() == 1) {
        rc.push_back(sm[0].str());
      }
      p = sm.suffix();
    }
    return rc;
  }

}  // namespace Dwm
