//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmpftbl.cc 8825 2017-03-20 19:36:22Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmpftbl.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
}

#include <algorithm>

#include "DwmDateTime.hh"
#include "DwmEngFormat.hh"
#include "DwmOptArgs.hh"
#include "DwmPfDevice.hh"

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void PrintTableEntryStat(ostream & os, const Pf::TableEntryStat & tes)
{
  if (os) {
    Ipv4Prefix  pfx(tes.Prefix());
    DateTime    lastCleared(tes.LastZeroed());
    os << setiosflags(ios::left)
       << setw(18) << pfx.ToShortString() << ' '
       << lastCleared.Formatted("%Y/%m/%d") << ' '
       << EngFormat(tes.InPktsBlocked()) << ' '
       << EngFormat(tes.InBytesBlocked()) << ' '
       << EngFormat(tes.InPktsPassed()) << ' '
       << EngFormat(tes.InBytesPassed()) << '\n'
       << resetiosflags(ios::left);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void PrintTableEntryStats(ostream & os, vector<Pf::TableEntryStat> & tes)
{
  if (os && (! tes.empty())) {
    sort(tes.begin(), tes.end(),
         [&] (const Pf::TableEntryStat & t1,
              const Pf::TableEntryStat & t2) 
         { return (t1.InPktsBlocked() > t2.InPktsBlocked()); });
    for (auto t : tes) {
      PrintTableEntryStat(cout, t);
    }
  }
  return;
}
      
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("a:", "anchor", false, "", "anchor");
  optargs.AddOptArg("D:", "delete_table", false, "false", "delete table");
  optargs.AddOptArg("d", "delete", false, "false", "delete prefix");
  optargs.AddOptArg("i", "insert", false, "false", "insert prefix");
  optargs.AddOptArg("g", "get", false, "false", "get prefixes");
  optargs.AddOptArg("l", "list", false, "false", "list tables");
  optargs.AddOptArg("p:", "prefix", false, "", "prefix");
  optargs.AddOptArg("s", "stats", false, "", "show statistics");
  optargs.AddOptArg("t:", "table", false, "", "table name");
  int  nextArg = optargs.Parse(argc, argv);

  if (((! optargs.Get<bool>("insert"))
       && (! optargs.Get<bool>("delete"))
       && (! optargs.Get<bool>("get"))
       && (! optargs.Get<bool>("list")))
      || (optargs.Get<string>("table").empty()
          && optargs.Get<bool>("list") == false)) {
    optargs.Usage(argv[0]);
    return 1;
  }

  int  rc = 1;
  
  Pf::Device  pfdev("/dev/pf");
  if (pfdev.Open()) {
    Pf::Table  &&table = pfdev.GetTable(optargs.Get<string>("anchor"),
                                        optargs.Get<string>("table"));
    if (optargs.Get<bool>("insert")) {
      if (! optargs.Get<string>("prefix").empty()) {
        if (table.Add(Ipv4Prefix(optargs.Get<string>("prefix")))) {
          rc = 0;
        }
      }
    }
    else if (optargs.Get<bool>("delete")) {
      if (! optargs.Get<string>("prefix").empty()) {
        if (table.Remove(Ipv4Prefix(optargs.Get<string>("prefix")))) {
          rc = 0;
        }
      }
    }
    else if (optargs.Get<bool>("get")) {
      if (optargs.Get<bool>("stats")) {
        vector<Pf::TableEntryStat>  &&stats = table.GetStats();
        if (! stats.empty()) {
          PrintTableEntryStats(cout, stats);
          rc = 0;
        }
      }
      else {
        vector<Ipv4Prefix>  &&prefixes = table.GetEntries();
        if (! prefixes.empty()) {
          rc = 0;
          for (auto p : prefixes) {
            cout << p.ToShortString() << '\n';
          }
        }
      }
    }
    else if (optargs.Get<bool>("list")) {
      vector<Pf::Table>  &&tables = pfdev.GetTables();
      if (! tables.empty()) {
        rc = 0;
        for (auto t : tables) {
          if (! t.Anchor().empty()) {
            cout << t.Anchor() << '/';
          }
          cout << t.Name() << '\n';
        }
      }
    }
    else if (! optargs.Get<string>("delete_table").empty()) {
      if (optargs.Get<string>("table").empty()) {
        optargs.Usage(argv[0]);
      }
      else {
        Pf::Table  && t = pfdev.GetTable(optargs.Get<string>("anchor"),
                                         optargs.Get<string>("table"));
        if (pfdev.TableExists(t)) {
          if (pfdev.DeleteTable(t.Anchor(),t.Name())) {
            rc = 0;
          }
        }
      }
    }
    
    pfdev.Close();
  }
  else {
    cerr << "Failed to open /dev/pf\n";
  }

  return rc;
}

  
