//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmPfDevice.cc 9415 2017-06-03 23:04:48Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPfDevice.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/ioctl.h>
  #include <fcntl.h>
  #include <unistd.h>
}

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmPfDevice.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmPf/tags/libDwmPf-0.1.6/src/DwmPfDevice.cc 9415 $");

using namespace std;

namespace Dwm {

  namespace Pf {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Device::Device(const string & path)
        : _path(path), _fd(-1), _mtx()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Device::~Device()
    {
      Close();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::Open()
    {
      Close();
      lock_guard<mutex>  lock(_mtx);
      _fd = open(_path.c_str(), O_RDWR);
      return (_fd >= 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Device::Close()
    {
      lock_guard<mutex>  lock(_mtx);
      if (_fd >= 0) {
        close(_fd);
        _fd = -1;
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::IsOpen() const
    {
      lock_guard<mutex>  lock(_mtx);
      return (_fd >= 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Table Device::GetTable(const std::string & anchor,
                           const std::string & name) const
    {
      vector<Table>  tables = GetTables();
      auto  i = find_if(tables.begin(), tables.end(),
                        [&] (const Table & t) {
                          return ((t.Name() == name)
                                  && (t.Anchor() == anchor));
                        });
      if (i != tables.end()) {
        return Table(*i);
      }
      else {
        return Table(*this, "","");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::vector<Table> Device::GetTables() const
    {
      vector<Table>  tables;
      if (_fd >= 0) {
        Table::IOCTable  ioctbl;
        if (ioctbl.AllocPfrTables(1)) {
          pfioc_table  *ioctblp = ioctbl.pfioc();
          if (Ioctl(DIOCRGETTABLES, ioctblp)) {
            int  numTables = ioctblp->pfrio_size;
            if (ioctbl.AllocPfrTables(numTables)) {
              if (Ioctl(DIOCRGETTABLES, ioctblp)) {
                pfr_table  *tbl = (pfr_table *)ioctblp->pfrio_buffer;
                for (int i = 0; i < numTables; ++i) {
                  tables.push_back(Table(*this,*tbl));
                  ++tbl;
                }
              }
            }
          }
        }
      }
      else {
        Syslog(LOG_ERR, "Pf::Device not open");
      }
      
      return tables;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::TableExists(const Table & table) const
    {
      bool  rc = false;
      if (_fd >= 0) {
        vector<Table> &&tables = GetTables();
        auto  i = find_if(tables.begin(), tables.end(),
                          [&] (const Table & t) {
                            return ((t.Name() == table.Name())
                                    && (t.Anchor() == table.Anchor()));
                          });
        if (i != tables.end()) {
          rc = true;
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::CreateTable(const string & anchor, const string & name) const
    {
      bool  rc = false;
      if (_fd >= 0) {
        Table  table(*this, anchor, name);
        if (! TableExists(table)) {
          Table::IOCTable  ioctbl(table);
          ioctbl.AllocPfrTables(1);
          pfioc_table  *ioctblp = ioctbl.pfioc();
          pfr_table  *newtbl = (pfr_table *)ioctblp->pfrio_buffer;
          strlcpy(newtbl->pfrt_anchor, anchor.c_str(),
                  sizeof(newtbl->pfrt_anchor));
          strlcpy(newtbl->pfrt_name, name.c_str(),
                  sizeof(newtbl->pfrt_name));
          newtbl->pfrt_flags = PFR_TFLAG_PERSIST;
          if (Ioctl(DIOCRADDTABLES, ioctblp)) {
            rc = true;
          }
        }
      }
      else {
        Syslog(LOG_ERR, "Pf::Device not open");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::DeleteTable(const string & anchor, const string & name) const
    {
      bool  rc = false;
      if (_fd >= 0) {
        Table  table(*this, anchor, name);
        if (TableExists(table)) {
          Table::IOCTable  ioctbl(table);
          ioctbl.AllocPfrTables(1);
          pfioc_table  *ioctblp = ioctbl.pfioc();
          pfr_table  *newtbl = (pfr_table *)ioctblp->pfrio_buffer;
          strlcpy(newtbl->pfrt_anchor, anchor.c_str(),
                  sizeof(newtbl->pfrt_anchor));
          strlcpy(newtbl->pfrt_name, name.c_str(),
                  sizeof(newtbl->pfrt_name));
          if (Ioctl(DIOCRDELTABLES, ioctblp)) {
            rc = true;
          }
        }
        else {
          rc = true;
        }
      }
      else {
        Syslog(LOG_ERR, "Pf::Device not open");
      }
      
      return rc;
    }

#ifndef __APPLE__
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<Rule> Device::GetRules() const
    {
      vector<Rule>  rules;
      if (_fd >= 0) {
        struct pfioc_rule  pr;
        memset(&pr, 0, sizeof(pr));
        // pr.rule.action = PF_PASS;
        if (Ioctl(DIOCGETRULES, &pr)) {
          uint32_t  numRules = pr.nr;
          for (uint32_t rn = 0; rn < numRules; ++rn) {
            rules.push_back(Rule(*this, pr, rn));
          }
        }
      }
      else {
        Syslog(LOG_ERR, "Pf::Device not open");
      }
      return rules;
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    vector<Rule> Device::GetRulesWithTable(const string & tableName) const
    {
      vector<Rule>  rules;
      if (_fd >= 0) {
        struct pfioc_rule  pr;
        memset(&pr, 0, sizeof(pr));
        if (Ioctl(DIOCGETRULES, &pr)) {
          uint32_t  numRules = pr.nr;
          for (uint32_t rn = 0; rn < numRules; ++rn) {
            pr.nr = rn;
            if (Ioctl(DIOCGETRULE, &pr)) {
              if (((pr.rule.src.addr.type == PF_ADDR_TABLE)
                   && (pr.rule.src.addr.v.tblname == tableName))
                  ||
                  ((pr.rule.dst.addr.type == PF_ADDR_TABLE)
                   && (pr.rule.dst.addr.v.tblname == tableName))) {
                rules.push_back(Rule(*this, pr.rule, rn));
              }
            }
          }
        }
      }
      else {
        Syslog(LOG_ERR, "Pf::Device not open");
      }
      return rules;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Device::Ioctl(unsigned long req, void *v) const
    {
      bool  rc = false;
      lock_guard<mutex>  lock(_mtx);
      if (_fd >= 0) {
        if (ioctl(_fd, req, v) != -1) {
          rc = true;
        }
        else {
          Syslog(LOG_ERR, "ioctl(%d,%s,%p) failed: %m",
                 _fd, IoctlReqName(req).c_str(), v);
        }
      }
      else {
        Syslog(LOG_ERR, "pf device is not open");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string Device::IoctlReqName(unsigned long r)
    {
      string  rc;
      auto i = _ioctlNames.find(r);
      if (i != _ioctlNames.end()) {
        rc = i->second;
      }
      else {
        rc = to_string(r);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const map<unsigned long, string>  Device::_ioctlNames = {
      { DIOCSTART,        "DIOCSTART" },
      { DIOCSTOP,         "DIOCSTOP" },
      { DIOCADDRULE,      "DIOCADDRULE" },
      { DIOCGETRULES,     "DIOCGETRULES" },
      { DIOCGETRULE,      "DIOCGETRULE" },
      { DIOCCLRSTATES,    "DIOCCLRSTATES" },
      { DIOCGETSTATE,     "DIOCGETSTATE" },
      { DIOCSETSTATUSIF,  "DIOCSETSTATUSIF" },
      { DIOCGETSTATUS,    "DIOCGETSTATUS" },
      { DIOCCLRSTATUS,    "DIOCCLRSTATUS" },
      { DIOCNATLOOK,      "DIOCNATLOOK" },
      { DIOCSETDEBUG,     "DIOCSETDEBUG" },
      { DIOCGETSTATES,    "DIOCGETSTATES" },
      { DIOCCHANGERULE,   "DIOCCHANGERULE" },
      { DIOCSETTIMEOUT,   "DIOCSETTIMEOUT" },
      { DIOCGETTIMEOUT,   "DIOCGETTIMEOUT" },
      { DIOCADDSTATE,     "DIOCADDSTATE" },
      { DIOCCLRRULECTRS,  "DIOCCLRRULECTRS" },
      { DIOCGETLIMIT,     "DIOCGETLIMIT" },
      { DIOCSETLIMIT,     "DIOCSETLIMIT" },
      { DIOCKILLSTATES,   "DIOCKILLSTATES" },
      { DIOCSTARTALTQ,    "DIOCSTARTALTQ" },
      { DIOCSTOPALTQ,     "DIOCSTOPALTQ" },
      { DIOCADDALTQ,      "DIOCADDALTQ" },
      { DIOCGETALTQS,     "DIOCGETALTQS" },
      { DIOCGETALTQ,      "DIOCGETALTQ" },
      { DIOCCHANGEALTQ,   "DIOCCHANGEALTQ" },
      { DIOCGETQSTATS,    "DIOCGETQSTATS" },
      { DIOCBEGINADDRS,   "DIOCBEGINADDRS" },
      { DIOCADDADDR,      "DIOCADDADDR" },
      { DIOCGETADDRS,     "DIOCGETADDRS" },
      { DIOCGETADDR,      "DIOCGETADDR" },
      { DIOCCHANGEADDR,   "DIOCCHANGEADDR" },
      { DIOCGETRULESETS,  "DIOCGETRULESETS" },
      { DIOCGETRULESET,   "DIOCGETRULESET" },
      { DIOCRCLRTABLES,   "DIOCRCLRTABLES" },
      { DIOCRADDTABLES,   "DIOCRADDTABLES" },
      { DIOCRDELTABLES,   "DIOCRDELTABLES" },
      { DIOCRGETTABLES,   "DIOCRGETTABLES" },
      { DIOCRGETTSTATS,   "DIOCRGETTSTATS" },
      { DIOCRCLRTSTATS,   "DIOCRCLRTSTATS" },
      { DIOCRCLRADDRS,    "DIOCRCLRADDRS" },
      { DIOCRADDADDRS,    "DIOCRADDADDRS" },
      { DIOCRDELADDRS,    "DIOCRDELADDRS" },
      { DIOCRSETADDRS,    "DIOCRSETADDRS" },
      { DIOCRGETADDRS,    "DIOCRGETADDRS" },
      { DIOCRGETASTATS,   "DIOCRGETASTATS" },
      { DIOCRCLRASTATS,   "DIOCRCLRASTATS" },
      { DIOCRTSTADDRS,    "DIOCRTSTADDRS" },
      { DIOCRSETTFLAGS,   "DIOCRSETTFLAGS" },
      { DIOCRINADEFINE,   "DIOCRINADEFINE" },
      { DIOCOSFPFLUSH,    "DIOCOSFPFLUSH" },
      { DIOCOSFPADD,      "DIOCOSFPADD" },
      { DIOCOSFPGET,      "DIOCOSFPGET" },
      { DIOCXBEGIN,       "DIOCXBEGIN" },
      { DIOCXCOMMIT,      "DIOCXCOMMIT" },
      { DIOCXROLLBACK,    "DIOCXROLLBACK" },
      { DIOCGETSRCNODES,  "DIOCGETSRCNODES" },
      { DIOCCLRSRCNODES,  "DIOCCLRSRCNODES" },
      { DIOCSETHOSTID,    "DIOCSETHOSTID" },
      { DIOCIGETIFACES,   "DIOCIGETIFACES" },
      { DIOCSETIFFLAG,    "DIOCSETIFFLAG" },
      { DIOCCLRIFFLAG,    "DIOCCLRIFFLAG" },
      { DIOCKILLSRCNODES, "DIOCKILLSRCNODES" },
      { DIOCGIFSPEED,     "DIOCGIFSPEED" }
    };

    
  }  // namespace Pf

}  // namespace Dwm
