//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPServer.hh 12125 2022-12-11 04:46:11Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPServer.hh
//!  \brief Dwm::RDAP::Server class definition
//---------------------------------------------------------------------------

#ifndef _DWMRDAPSERVER_HH_
#define _DWMRDAPSERVER_HH_

#include <chrono>
#include <thread>
#include <vector>

#include "DwmSocket.hh"
#include "DwmCredencePeer.hh"
#include "DwmIpv4CountryDb.hh"
#include "DwmRDAPResponder.hh"
#include "DwmRDAPServerConfig.hh"

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  Encapsulates my RDAP-lite server.  This is just the listener part,
    //!  each client will get its own Responder.  This class also holds the
    //!  database.
    //------------------------------------------------------------------------
    class Server
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from an executor, path to my Credence key directory and
      //!  a server configuration object.
      //----------------------------------------------------------------------
      template <typename Ex>
      Server(Ex executor, const std::string & keyPath,
             const ServerConfig & config)
          : _keyStash(keyPath), _knownKeys(keyPath), _config(config),
            _db(config.DatabaseFile()), _cc(), _acceptors(), _responders()
      {
        _cc.ReadJson(_config.CountryCodesFile());
        _db.AggregateAdjacents();

        namespace ip = boost::asio::ip;
        using boost::asio::ip::tcp;

        ip::tcp::acceptor  acc(executor);
        Ipv4Address  bindAddr = _config.TCPAddress();
        if (bindAddr.Raw() == INADDR_ANY) {
          tcp::endpoint  ep(tcp::v4(), _config.TCPPort());
          acc.open(tcp::v4());
          acc.bind(ep);
        }
        else {
          ip::address  addr = ip::address::from_string((std::string)bindAddr);
          tcp::endpoint  ep(addr, _config.TCPPort());
          acc.open(tcp::v4());
          acc.bind(ep);
        }
        
        auto & a = _acceptors.emplace_back(std::move(acc));
        a.listen();
        AcceptLoop(a);
      }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ~Server()
      {
        Syslog(LOG_INFO, "Server destroyed");
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      const Credence::KeyStash & GetKeyStash() const  { return _keyStash; }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      const Credence::KnownKeys & GetKnownKeys() const  { return _knownKeys; }

      Ipv4CountryDb & GetDb()    { return _db; }
      
      //----------------------------------------------------------------------
      //!  Start the server.
      //----------------------------------------------------------------------
      bool Start();
      
      //----------------------------------------------------------------------
      //!  Stop the server.
      //----------------------------------------------------------------------
      void Stop();

    private:
      Credence::KeyStash                          _keyStash;
      Credence::KnownKeys                         _knownKeys;
      const ServerConfig                         &_config;
      Ipv4CountryDb                               _db;
      CountryCodes                                _cc;
      std::deque<boost::asio::ip::tcp::acceptor>  _acceptors;
      std::vector<std::shared_ptr<Responder>>     _responders;

      void CleanupResponders();
      void AcceptLoop(boost::asio::ip::tcp::acceptor & a);
    };
    
  }  // namespace RDAP

}  // namespace Dwm

#endif  // _DWMRDAPSERVER_HH_
