//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPQuery.hh 12077 2022-11-25 18:56:56Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPQuery.hh
//!  \brief Dwm::RDAP query class definitions
//---------------------------------------------------------------------------

#ifndef _DWMRDAPQUERY_HH_
#define _DWMRDAPQUERY_HH_

#include <string>
#include <nlohmann/json.hpp>

#include "DwmIpv4Address.hh"
#include "DwmRDAPResponse.hh"

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  Base RDAP query class.  Not used directly; use one of the classes
    //!  that inherits from this class.
    //------------------------------------------------------------------------
    class Query
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      Query();

      const std::string & Uri() const;
      
      //----------------------------------------------------------------------
      //!  Executes the query and returns the response.
      //----------------------------------------------------------------------
      Response Execute() const;
      
      //----------------------------------------------------------------------
      //!  <rant>
      //!  Some of the registries (ARIN, for example) return IPv4 addresses
      //!  with leading zeros in each octet in their RDAP responses.  This is
      //!  an incredibly stupid thing to do since it causes address
      //!  conversion routines like inet_addr(), inet_aton(), et. al. to
      //!  interpret these octets as octal (as they should!) and hence yield
      //!  incorrect results.  inet_pton() will fail to parse addresses in
      //!  this format.  So basically, the addresses returned by some of the
      //!  registry RDAP servers are completely unusable by machines without
      //!  massaging the addresses.  So much for RDAP being machine-friendly.
      //!  </rant>
      //!  This function is used to fix the 'startAddress' and 'endAddress'
      //!  fields in an RDAP JSON response for an ip query.  It should likely
      //!  be used for more fields since I'm sure the stupidity isn't
      //!  restricted to just these fields.
      //----------------------------------------------------------------------
      static void FixIpv4Stupidity(nlohmann::json & jv);
      
    protected:
      std::string  _uristr;
    };

    //------------------------------------------------------------------------
    //!  Encapsulates an IPv4 RDAP query (rdap/ip/a.b.c.d).
    //------------------------------------------------------------------------
    class IPv4Query : public Query
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      IPv4Query(const Ipv4Address & ip);

      IPv4Response Execute() const;
    };

    //------------------------------------------------------------------------
    //!  NOT IMPLEMENTED, NO BOOTSTRAP AVAILABLE
    //------------------------------------------------------------------------
    class ASQuery : public Query
    {
    public:
      ASQuery(uint32_t asnum);
    };

    //------------------------------------------------------------------------
    //!  NOT IMPLEMENTED, NO BOOTSTRAP AVAILABLE
    //------------------------------------------------------------------------
    class DomainQuery : public Query
    {
    public:
      DomainQuery(const std::string & domain);
    };
    
  }  // namespace RDAP

}  // namespace Dwm

#endif  // _DWMRDAPQUERY_HH_
