//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockRequestMessage.hh 12351 2024-05-03 09:12:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockRequestMessage.hh
//!  \brief Dwm::McBlock::RequestMessage class definition
//---------------------------------------------------------------------------

#include <string>
#include <vector>
#include <nlohmann/json.hpp>

#include "DwmIpv4Prefix.hh"
#include "DwmTimeValue64.hh"
#include "DwmCredencePeer.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  Convenience class to create an 'add' request for mcblockd.
    //------------------------------------------------------------------------
    class AddReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct an 'add' request for the given pf @c table, @c country,
      //!  @c days and @c prefixes.
      //----------------------------------------------------------------------
      AddReq(const std::string & table,
             const std::string & country,
             uint32_t days,
             const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'delete' request for mcblockd.
    //------------------------------------------------------------------------
    class DeleteReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct an 'add' request for the given pf @c table and
      //!  @c prefixes.
      //----------------------------------------------------------------------
      DeleteReq(const std::string & table,
                const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create an 'activate' request for mcblockd.
    //------------------------------------------------------------------------
    class ActivateReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct an 'activate' request for the given pf @c table and
      //!  given @c prefixes.
      //----------------------------------------------------------------------
      ActivateReq(const std::string & table,
                  const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'deactivate' request for mcblockd.
    //------------------------------------------------------------------------
    class DeactivateReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'deactivate' request for the given pf @c table and
      //!  given @c prefix.
      //----------------------------------------------------------------------
      DeactivateReq(const std::string & table,
                    const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'search' request for mcblockd.
    //------------------------------------------------------------------------
    class SearchReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'search' request for the given @c addr.
      //----------------------------------------------------------------------
      SearchReq(const Ipv4Address & addr);
    };
    
    //------------------------------------------------------------------------
    //!  Convenience class to create a 'getActive' request for mcblockd.
    //------------------------------------------------------------------------
    class GetActiveReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'getActive' request for the given pf @c table.
      //----------------------------------------------------------------------
      GetActiveReq(const std::string & table);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'getAddRules' request for mcblockd.
    //------------------------------------------------------------------------
    class GetAddRulesReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'getAddRules' request for the given pf @c table.
      //----------------------------------------------------------------------
      GetAddRulesReq(const std::string & table);
    };
    
    //------------------------------------------------------------------------
    //!  NOT IMPLEMENTED
    //------------------------------------------------------------------------
    class GetMatchesReq : public nlohmann::json
    {
    public:
      GetMatchesReq(const std::string & table, const Ipv4Prefix & prefix);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'logHit' request for mcblockd.
    //------------------------------------------------------------------------
    class LogHitReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'logHit' request for the given pf @c table, given
      //!  @c addr and given log timestamp @c logTime.
      //----------------------------------------------------------------------
      LogHitReq(const std::string & table, const Ipv4Address & addr,
                const TimeValue64 & logTime);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'get' request for mcblockd.
    //------------------------------------------------------------------------
    class GetReq : public nlohmann::json
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'get' request for the given pf @c table, given
      //!  @c addr.
      //----------------------------------------------------------------------
      GetReq(const std::string & table, const Ipv4Address & addr);
    };
    
    //------------------------------------------------------------------------
    //!  Encapsulates a request message for mcblockd.
    //------------------------------------------------------------------------
    class RequestMessage
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      RequestMessage();
      
      //----------------------------------------------------------------------
      //!  Construct from a nlohmann::Json reference
      //----------------------------------------------------------------------
      RequestMessage(const nlohmann::json & req);
      
      //----------------------------------------------------------------------
      //!  Construct from a nlohmann::json rvalue reference.
      //----------------------------------------------------------------------
      RequestMessage(const nlohmann::json && req);

      //----------------------------------------------------------------------
      //!  Reads the message from the given @c peer.  Returns true on success,
      //!  false on failure.
      //----------------------------------------------------------------------
      bool Read(Credence::Peer & peer);
      
      //----------------------------------------------------------------------
      //!  Writes the message to the given @c peer.  Returns true on success,
      //!  false on failure.
      //----------------------------------------------------------------------
      bool Write(Credence::Peer & peer) const;

      //----------------------------------------------------------------------
      //!  Returns a const reference to the contained nlohmann::json
      //!  representation of the message.
      //----------------------------------------------------------------------
      const nlohmann::json & Json() const;
      
    private:
      nlohmann::json  _value;
    };
    
  }  // namespace McBlock

}  // namespace Dwm
