//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockServerConfig.cc 12091 2022-11-29 05:05:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockServerConfig.cc
//!  \brief Dwm::McBlock::ServerConfig class implementation
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmMcBlockServerConfig.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.3.7/classes/src/DwmMcBlockServerConfig.cc 12091 $");

using namespace std;

namespace Dwm {

  namespace McBlock {
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ServerConfig::ServerConfig(const Cfg2Json & config)
        : _pfdevpath("/dev/pf"), _tcpAddr(INADDR_ANY), _tcpPort(1001),
          _mcastAddr("224.0.0.42"), _mcastPort(1001),
          _keyDirectory("/usr/local/etc/mcblockd"),
          _dwmRDAPServer("127.0.0.1"), _databaseDirectory("/etc")
    {
      const nlohmann::json  *jvp = config.Find("Server");
      if (jvp) {
        const nlohmann::json  *p = config.Find(jvp, "PF/Device");
        if (p && p->is_string()) {
          _pfdevpath = p->get<string>();
        }
        p = config.Find(jvp, "TCP/Address");
        if (p && p->is_string()) {
          _tcpAddr = p->get<string>();
        }
        p = config.Find(jvp, "TCP/Port");
        if (p && p->is_string()) {
          _tcpPort = stoul(p->get<string>());
        }
        p = config.Find("Multicast/Address");
        if (p && p->is_string()) {
          _mcastAddr = p->get<string>();
        }
        p = config.Find(jvp, "Multicast/Port");
        if (p && p->is_string()) {
          _mcastPort = stoul(p->get<string>());
        }
        p = config.Find(jvp, "Auth/KeyDirectory");
        if (p && p->is_string()) {
          _keyDirectory = p->get<string>();
        }
        p = config.Find(jvp, "DwmRDAP/Server");
        if (p && p->is_string()) {
          _dwmRDAPServer = p->get<string>();
        }
        p = config.Find(jvp, "DatabaseDirectory");
        if (p && p->is_string()) {
          _databaseDirectory = p->get<string>();
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ServerConfig::Device() const
    {
      return _pfdevpath;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & ServerConfig::Device(const std::string & path)
    {
      _pfdevpath = path;
      return _pfdevpath;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::TCPAddr() const
    {
      return _tcpAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::TCPAddr(const Ipv4Address & addr)
    {
      _tcpAddr = addr;
      return _tcpAddr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::TCPPort() const
    {
      return _tcpPort;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::TCPPort(uint16_t tcpPort)
    {
      _tcpPort = tcpPort;
      return _tcpPort;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & ServerConfig::MulticastAddr() const
    {
      return _mcastAddr;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address &
    ServerConfig::MulticastAddr(const Ipv4Address & addr)
    {
      _mcastAddr = addr;
      return _mcastAddr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::MulticastPort() const
    {
      return _mcastPort;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t ServerConfig::MulticastPort(uint16_t port)
    {
      _mcastPort = port;
      return _mcastPort;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::KeyDirectory() const
    {
      return _keyDirectory;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::DwmRDAPServer() const
    {
      return _dwmRDAPServer;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & ServerConfig::DatabaseDirectory() const
    {
      return _databaseDirectory;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json ServerConfig::Json() const
    {
      nlohmann::json  value;
      value["pfdevpath"] = _pfdevpath;
      value["tcpAddr"] = (string)_tcpAddr;
      value["tcpPort"] = _tcpPort;
      value["mcastAddr"] = (string)_mcastAddr;
      value["mcastPort"] = _mcastPort;
      value["dwmRDAPServer"] = _dwmRDAPServer;
      value["databaseDirectory"] = _databaseDirectory;
      return value;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json ServerConfig::Json(const nlohmann::json & value)
    {
      _pfdevpath = value["pfdevpath"].get<string>();
      _tcpAddr = Ipv4Address(value["tcpAddr"].get<string>());
      _tcpPort = value["tcpPort"].get<uint16_t>();
      _mcastAddr = Ipv4Address(value["mcastAddr"].get<string>());
      _mcastPort = value["mcastPort"].get<uint16_t>();
      _dwmRDAPServer = value["dwmRDAPServer"].get<string>();
      return Json();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool ServerConfig::operator == (const ServerConfig & config) const
    {
      return ((_pfdevpath == config._pfdevpath)
              && (_tcpAddr == config._tcpAddr)
              && (_tcpPort == config._tcpPort)
              && (_mcastAddr == config._mcastAddr)
              && (_mcastPort == config._mcastPort)
              && (_keyDirectory == config._keyDirectory)
              && (_dwmRDAPServer == config._dwmRDAPServer)
              && (_databaseDirectory == config._databaseDirectory));
    }
    
  }  // namespace McBlock

}  // namespace Dwm
