//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/apps/mcroverd/mcroverd.cc 12323 $
// @(#) $Id: mcroverd.cc 12323 2024-01-14 05:31:56Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file httptest.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
  #include <sys/types.h>
  #include <sys/time.h>
  #include <sys/resource.h>
}

#include <array>
#include <atomic>
#include <chrono>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <thread>

#include "DwmDaemonUtils.hh"
#include "DwmSignal.hh"
#include "DwmIO.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverConfig.hh"
#include "DwmMcroverBackupUtils.hh"
#include "DwmMcroverBanditUtils.hh"
#include "DwmMcroverCredencePeerUtils.hh"
#include "DwmMcroverDiskUtils.hh"
#include "DwmMcroverDNSUtils.hh"
#include "DwmMcroverFilesystemUtils.hh"
#include "DwmMcroverGuestUtils.hh"
#include "DwmMcroverHTTPUtils.hh"
#include "DwmMcroverRouteUtils.hh"
#include "DwmMcroverRPCUtils.hh"
#include "DwmMcroverSMTPUtils.hh"
#include "DwmMcroverTcp4Utils.hh"
#include "DwmMcroverTcp6Utils.hh"
#include "DwmMcroverUPSUtils.hh"
#include "DwmMcroverWebAppUtils.hh"
#include "DwmMcroverZFSUtils.hh"
#include "DwmMcroverUtils.hh"
#include "DwmMcroverServer.hh"
#include "DwmMcroverPackMember.hh"

using namespace std;
using namespace Dwm;

using Mcrover::Utils, Mcrover::BackupUtils, Mcrover::BanditUtils,
  Mcrover::CredencePeerUtils, Mcrover::DiskUtils, Mcrover::DNSUtils,
  Mcrover::FilesystemUtils, Mcrover::GuestUtils, Mcrover::HTTPUtils,
  Mcrover::RouteUtils, Mcrover::RPCUtils, Mcrover::SMTPUtils,
  Mcrover::Tcp4Utils, Mcrover::Tcp6Utils, Mcrover::UPSUtils,
  Mcrover::WebAppUtils, Mcrover::ZFSUtils, Mcrover::Config,
  Mcrover::LocalHostConfig, Mcrover::TargetHostConfig, Mcrover::PackMember,
  Mcrover::AlertOrigin, Mcrover::AlertBowl, Mcrover::PackMemberConfig;

static atomic<bool>  g_needConfigReload = false;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SaveAlerts(const AlertBowl & alertBowl, const string & fileName) 
{
  if (! alertBowl.Save(fileName)) {
    Syslog(LOG_ERR, "Failed to save alerts to '%s'", fileName.c_str());
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool RestoreAlerts(AlertBowl & alertBowl, const string & fileName)
{
  return alertBowl.Load(fileName);
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetPackMemberAlerts(const Config & config,
                                const PackMemberConfig & memberConfig,
                                AlertBowl & packAlerts)
{
  using Dwm::Mcrover::RoverAlert;
    
  PackMember  member(memberConfig, config.Service().KeyDirectory());
  AlertBowl                memberAlerts;
  RoverAlert::FailureType  failure;
  if (member.GetMemberAlerts(memberAlerts, failure)) {
    packAlerts.Add(memberAlerts);
  }
  else {
    AlertOrigin  myOrigin = config.MyOrigin();
    IpAddress    memberAddr(memberConfig.Addresses().begin()->address().to_string());
    packAlerts.Add(myOrigin, RoverAlert(memberAddr, failure));
    Syslog(LOG_ERR, "Failed to get alerts from %s",
           ((string)memberAddr).c_str());
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetPackAlerts(const Config & config, AlertBowl & packAlerts)
{
  vector<pair<string,thread>>  threads;
  for (const auto & memberConfig : config.MyPack().Members()) {
    threads.emplace_back(make_pair(memberConfig.second.Name(),
                                   thread([&] {
                                     GetPackMemberAlerts(std::ref(config),
                                                         std::ref(memberConfig.second),
                                                         std::ref(packAlerts));
                                   })));
  }
  for (auto & memberThread : threads) {
    if (memberThread.second.joinable()) {
      memberThread.second.join();
      Syslog(LOG_INFO, "Pack member thread joined for %s",
             memberThread.first.c_str());
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetAllAlerts(const Config & config,
                         const vector<TargetHostConfig> & hosts,
                         AlertBowl & alerts)
{
  AlertOrigin  origin = config.MyOrigin();
  
  array<AlertBowl,9>   tmp;
  thread         threads[9] = {
    thread([&]{ DNSUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                    std::ref(tmp[0])); }),
    thread([&]{ RPCUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                    std::ref(tmp[1])); }),
    thread([&]{ Tcp4Utils::GetAlerts(std::ref(origin), std::ref(hosts),
                                     std::ref(tmp[2])); }),
    thread([&]{ HTTPUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                     std::ref(tmp[3])); }),
    thread([&]{ SMTPUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                     std::ref(tmp[4])); }),
    thread([&]{ UPSUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                    std::ref(tmp[5])); }),
    thread([&]{ Tcp6Utils::GetAlerts(std::ref(origin), std::ref(hosts),
                                     std::ref(tmp[6])); }),
    thread([&]{ WebAppUtils::GetAlerts(std::ref(origin), std::ref(hosts),
                                       std::ref(tmp[7])); }),
    thread([&]{ CredencePeerUtils::GetAlerts(std::ref(origin),
                                             std::ref(config),
                                             std::ref(tmp[8])); })
  };

  for (int i = 0; i < 9; ++i) {
    if (threads[i].joinable()) {
      threads[i].join();
    }
    if (! tmp[i].Empty()) {
      alerts.Add(tmp[i]);
    }
  }

  //  these are all local and should be non-blocking, so no need for threads.
  ZFSUtils::GetAlerts(origin, config.Local(), alerts);
  RouteUtils::GetAlerts(origin, config.Local(), alerts);
  GuestUtils::GetAlerts(origin, config.Local(), alerts);
  BanditUtils::GetAlerts(origin, config.Local(), alerts);
  FilesystemUtils::GetAlerts(origin, config, alerts);
  DiskUtils::GetAlerts(origin, config.Local(), alerts);
  BackupUtils::GetAlerts(origin, config.Local(), alerts);
  
  GetPackAlerts(config, alerts);
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SigHupHandler(int sigNum)
{
  if (SIGHUP == sigNum) {
    Syslog(LOG_INFO, "SIGHUP received.");
    g_needConfigReload = true;
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SavePID(const string & pidFile)
{
#ifndef O_EXLOCK
  #define O_EXLOCK 0
#endif
  pid_t  pid = getpid();
  int    fd = open(pidFile.c_str(), O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644);
  if (fd >= 0) {
    string  &&pidstr = to_string(pid) + "\n";
    write(fd, pidstr.c_str(), pidstr.size());
    close(fd);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const char *argv0)
{
  std::cerr << "Usage: " << argv0 << " [-d] [-c configFile] [-p pidFile]\n\n"
            << "-d: don't daemonize (run in foreground)\n"
            << "default configFile: /usr/local/etc/mcroverd.cfg\n"
            << "default pidFile: /var/run/mcroverd.pid\n";
  return;
}
  
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int  rc = 1;
  string  cfgPath("/usr/local/etc/mcroverd.cfg");
  string  pidFile("/var/run/mcroverd.pid");
  bool    detach = true;
  
  int  optChar;
  while ((optChar = getopt(argc, argv, "c:dp:")) != -1) {
    switch (optChar) {
      case 'c':
        cfgPath = optarg;
        break;
      case 'd':
        detach = false;
        break;
      case 'p':
        pidFile = optarg;
        break;
      default:
        std::cerr << "Invalid argument '" << (char)optChar << "'\n";
        Usage(argv[0]);
        exit(1);
        break;
    }
  }

  Config  config;
  if (config.Parse(cfgPath)) {
    if (detach) {
      DaemonUtils::Daemonize();
      close(STDERR_FILENO);
    }
    Dwm::SysLogger::Open("mcroverd", LOG_PID, config.SyslogFacility());
    Dwm::SysLogger::MinimumPriority(config.SyslogLevel());
    Dwm::SysLogger::ShowFileLocation(config.SyslogLocations());
    
    Syslog(LOG_INFO, "Started.  %lu pack members, %lu servers",
           config.MyPack().Members().size(), config.Servers().size());
    
    Signal sigHup(SIGHUP);
    sigHup.PushHandler(SigHupHandler);

    SavePID(pidFile);

    AlertBowl  alertBowl;
    alertBowl.Load(config.AlertFile());

    shared_ptr<Mcrover::Server>  server;
    if (config.RunService()) {
      server = make_shared<Mcrover::Server>(config);
      server->Start();
    }
    time_t  nextPollTime = 0;
    for (;;) {
      if (g_needConfigReload) {
        Config  newConfig;
        if (newConfig.Parse(cfgPath)) {
          config = newConfig;
          Dwm::SysLogger::Close();
          Dwm::SysLogger::Open("mcroverd", LOG_PID, config.SyslogFacility());
          Dwm::SysLogger::MinimumPriority(config.SyslogLevel());
          Dwm::SysLogger::ShowFileLocation(config.SyslogLocations());
          if (server) {
            server->Stop();
            server = nullptr;
          }
          if (config.RunService()) {
            server = make_shared<Mcrover::Server>(config);
            server->Start();
          }
          Syslog(LOG_INFO, "Configuration reloaded.");
        }
        else {
          Syslog(LOG_ERR, "Invalid configuration; keeping old one.");
        }
        g_needConfigReload = false;
      }
      time_t  now = time((time_t *)0);
      if (now < nextPollTime) {
        this_thread::sleep_for((nextPollTime - now) < 5
                               ? chrono::seconds(nextPollTime - now)
                               : chrono::seconds(5));
      }
      else {
        chrono::steady_clock::time_point  pollStart =
          chrono::steady_clock::now();
        AlertBowl  alerts;
        GetAllAlerts(config, config.Servers(), alerts);
        alertBowl.Refresh(alerts);
        SaveAlerts(alertBowl, config.AlertFile());
        chrono::steady_clock::time_point  pollEnd =
          chrono::steady_clock::now();
        chrono::duration<long,milli>  pollPeriod =
          chrono::duration_cast<chrono::milliseconds>(pollEnd - pollStart);
        
        Syslog(LOG_INFO, "Poll finished in %gs", pollPeriod.count() / 1000.0);
        
        nextPollTime = now + 60;
        struct rusage  rusage;
        getrusage(RUSAGE_SELF, &rusage);
        Syslog(LOG_INFO, "maxrss %ldM", rusage.ru_maxrss / 1024);
      }
    }
  }
  else {
    cerr << "Failed to parse config in '" << cfgPath << "'\n";
  }

  return rc;
}
