//===========================================================================
// @(#) $DwmPath$
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2022
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverCredencePeerAlert.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::CredencePeerAlert class implementation
//---------------------------------------------------------------------------

#include <map>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"

#include "DwmMcroverCredencePeerAlert.hh"

namespace Dwm {

  namespace Mcrover {

    using namespace std;

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    CredencePeerAlert::CredencePeerAlert()
        : _addr(), _step(e_stepNone)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    CredencePeerAlert::CredencePeerAlert(const TcpDstAddr & addr,
                                         StepEnum step)
        : _addr(addr), _step(step)
    { }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string CredencePeerAlert::StepName() const
    {
      static const map<uint8_t,string>  k_stepNames = {
        { e_stepNone,         "NONE"      },
        { e_stepConnect,      "CONN"      },
        { e_stepGetMyKeys,    "KEYSTASH"  },
        { e_stepGetKnownKeys, "KNOWNKEYS" },
        { e_stepAuthenticate, "AUTH"      }
      };
      string  rc = "UNKNOWN";
      auto    it = k_stepNames.find(_step);
      if (k_stepNames.end() != it) {
        rc = it->second;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string CredencePeerAlert::DisplayString() const
    {
      string  rc = (string)(_addr.Address()) + ':' + to_string(_addr.Port())
        + '(' + _addr.Name() + "):" + StepName() + " CREDENCE ";
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json CredencePeerAlert::ToJson() const
    {
      nlohmann::json  j;
      j["addr"] = (string)(_addr.Address());
      j["port"] = _addr.Port();
      j["name"] = _addr.Name();
      j["step"] = (uint16_t)_step;
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool CredencePeerAlert::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto addr = j.find("addr");
        if ((addr != j.end()) && addr->is_string()) {
          _addr.Address(IpAddress(addr->get<string>()));
          auto port = j.find("port");
          if ((port != j.end()) && port->is_number()) {
            _addr.Port(port->get<uint16_t>());
            auto name = j.find("name");
            if ((name != j.end()) && name->is_string()) {
              _addr.Name(name->get<string>());
              auto step = j.find("step");
              if ((step != j.end()) && step->is_number()) {
                _step = (StepEnum)step->get<uint16_t>();
                rc = true;
              }
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    istream & CredencePeerAlert::Read(istream & is)
    {
      if (is) {
        if (_addr.Read(is)) {
          StreamIO::Read(is, _step);
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & CredencePeerAlert::Write(ostream & os) const
    {
      if (os) {
        if (_addr.Write(os)) {
          StreamIO::Write(os, _step);
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t CredencePeerAlert::Read(int fd)
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        ssize_t  bytesRead = _addr.Read(fd);
        if (bytesRead > 0) {
          rc = bytesRead;
          bytesRead = DescriptorIO::Read(fd, _step);
          if (1 == bytesRead) {
            rc += bytesRead;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
          
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t CredencePeerAlert::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        ssize_t  bytesWritten = _addr.Write(fd);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          bytesWritten = DescriptorIO::Write(fd, _step);
          if (1 == bytesWritten) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t CredencePeerAlert::Read(FILE *f)
    {
      size_t  rc = 0;
      if (f) {
        if (_addr.Read(f)) {
          if (FileIO::Read(f, _step) == 1) {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t CredencePeerAlert::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (_addr.Write(f)) {
          if (FileIO::Write(f, _step) == 1) {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t CredencePeerAlert::StreamedLength() const
    {
      return (_addr.StreamedLength() + IOUtils::StreamedLength(_step));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool CredencePeerAlert::operator == (const CredencePeerAlert & alert) const
    {
      return ((_addr == alert._addr)
              && (_step == alert._step));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool CredencePeerAlert::operator < (const CredencePeerAlert & alert) const
    {
      bool  rc = false;
      if (_addr < alert._addr) {
        rc = true;
      }
      else {
        if (_addr == alert._addr) {
          rc = (_step < alert._step);
        }
      }
      return rc;
    }
    
  }  // namespace Mcrover

}  // namespace Dwm
