//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCProgramId.cc 11886 $
// @(#) $Id: DwmMcroverRPCProgramId.cc 11886 2021-04-07 05:16:39Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverRPCProgramId.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::RPCProgramId class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/uio.h>
  #include <rpc/rpc.h>
  #include <netdb.h>
}

#include <stdexcept>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmIOUtils.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverRPCProgramId.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCProgramId.cc 11886 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCProgramId::RPCProgramId(const nlohmann::json & j)
        : _progNum(0), _progName(""), _protocol(0), _version(0)

    {
      if (j.is_object()) {
        auto  name = j.find("name");
        if ((name != j.end()) && name->is_string()) {
          _progName = name->get<string>();
          auto  rpcEntry = getrpcbyname(_progName.c_str());
          if (rpcEntry) {
            _progNum = rpcEntry->r_number;
          }
          else if (_progName == "rpcbind") {
            rpcEntry = getrpcbyname("portmapper");
            if (rpcEntry) {
              _progNum = rpcEntry->r_number;
            }
          }
          if (rpcEntry) {
            auto  protocol = j.find("protocol");
            if ((protocol != j.end()) && protocol->is_string()) {
              auto  protoEntry =
                getprotobyname(protocol->get<string>().c_str());
              if (protoEntry) {
                _protocol = protoEntry->p_proto;
                auto  version = j.find("version");
                if ((version != j.end()) && version->is_number()) {
                  _version = version->get<unsigned long>();
                }
              }
            }
          }
        }
      }
      if ((! _progNum) || (_progName.empty()) || (! _protocol)
          || (! _version)) {
        throw std::invalid_argument("JSON not valid for RPCProgramID");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCProgramId::RPCProgramId(const string & progName,
                               unsigned long protocol,
                               unsigned long version)
        : _progName(progName), _protocol(protocol),
          _version(version)
    {
      auto  rpcEntry = getrpcbyname(progName.c_str());
      if (rpcEntry) {
        _progNum = rpcEntry->r_number;
      }
      else if (progName == "rpcbind") {
        rpcEntry = getrpcbyname("portmapper");
        if (rpcEntry) {
          _progNum = rpcEntry->r_number;
        }
        else {
          throw std::invalid_argument("Can't find RPC number for " + _progName);
        }
      }
      else {
        throw std::invalid_argument("Can't find RPC number for " + _progName);
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCProgramId::RPCProgramId(unsigned long progNum, unsigned long protocol,
                               unsigned long version)
        : _progNum(progNum), _progName(""), _protocol(protocol),
          _version(version)
    {
      auto  rpcEntry = getrpcbynumber(_progNum);
      if (rpcEntry) {
        _progName = rpcEntry->r_name;
      }
      else {
        Syslog(LOG_ERR, "Failed to find RPC name for program %ul", _progNum);
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RPCProgramId::RPCProgramId(unsigned long progNum,
                               const string & progName,
                               unsigned long protocol, unsigned long version)
        : _progNum(progNum), _progName(progName), _protocol(protocol),
          _version(version)
    { }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Number() const
    {
      return _progNum;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Number(uint32_t num)
    {
      _progNum = num;
      return _progNum;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RPCProgramId::Name() const
    {
      return _progName;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string RPCProgramId::Name(const string & name)
    {
      _progName = name;
      return _progName;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Protocol() const
    {
      return _protocol;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Protocol(uint32_t protocol)
    {
      _protocol = protocol;
      return _protocol;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Version() const
    {
      return _version;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t RPCProgramId::Version(uint32_t version)
    {
      _version = version;
      return _version;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCProgramId::operator < (const RPCProgramId & rpcpm) const
    {
      bool rc = false;
      if (_progNum < rpcpm._progNum) {
        rc = true;
      }
      else if (_progNum == rpcpm._progNum) {
        if (_protocol < rpcpm._protocol) {
          rc = true;
        }
        else if (_protocol == rpcpm._protocol) {
          if (_version < rpcpm._version) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCProgramId::operator == (const RPCProgramId & rpcpm) const
    {
      return ((_progNum == rpcpm._progNum)
              && (_protocol == rpcpm._protocol)
              && (_version == rpcpm._version));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RPCProgramId & rpcpm)
    {
      if (os) {
        auto  protoEntry = getprotobynumber(rpcpm._protocol);
        os << rpcpm._progNum << '(' << rpcpm._progName << ") ";
        if (protoEntry && protoEntry->p_name) {
          os << protoEntry->p_name;
        }
        else {
          os << rpcpm._protocol;
        }
        os << " version " << rpcpm._version;
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    istream & RPCProgramId::Read(istream & is)
    {
      if (is) {
        if (Dwm::StreamIO::Read(is, _progNum)) {
          if (Dwm::StreamIO::Read(is, _protocol)) {
            if (Dwm::StreamIO::Read(is, _version)) {
              Dwm::StreamIO::Read(is, _progName);
            }
          }
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RPCProgramId::Read(int fd)
    {
      ssize_t  rc = -1;
      static constexpr size_t  fixedSize =
        sizeof(_progNum) + sizeof(_protocol) + sizeof(_version);
      if (0 <= fd) {
        struct iovec  iov[3] = {
          { &_progNum, sizeof(_progNum) },
          { &_protocol, sizeof(_protocol) },
          { &_version, sizeof(_version) }
        };
        if (readv(fd, iov, 3) == fixedSize) {
          size_t  nameBytes = Dwm::DescriptorIO::Read(fd, _progName);
          if (nameBytes > 0) {
            _progNum = ntohl(_progNum);
            _protocol = ntohl(_protocol);
            _version = ntohl(_version);
            rc = fixedSize + nameBytes;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RPCProgramId::Read(FILE * f)
    {
      size_t  rc = 0;
      if (f) {
        if (Dwm::FileIO::Read(f, _progNum)) {
          if (Dwm::FileIO::Read(f, _protocol)) {
            if (Dwm::FileIO::Read(f, _version)) {
              rc = Dwm::FileIO::Read(f, _progName);
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & RPCProgramId::Write(ostream & os) const
    {
      if (os) {
        if (Dwm::StreamIO::Write(os, _progNum)) {
          if (Dwm::StreamIO::Write(os, _protocol)) {
            if (Dwm::StreamIO::Write(os, _version)) {
              Dwm::StreamIO::Write(os, _progName);
            }
          }
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t RPCProgramId::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        static constexpr size_t  fixedSize =
          sizeof(_progNum) + sizeof(_protocol) + sizeof(_version);
        decltype(_progNum)   progNum = htonl(_progNum);
        decltype(_protocol)  protocol = htonl(_protocol);
        decltype(_version)   version = htonl(_version);
        struct iovec  iov[3] = {
          { &progNum,  sizeof(progNum) },
          { &protocol, sizeof(protocol) },
          { &version,  sizeof(version) }
        };
        if (writev(fd, iov, 3) == fixedSize) {
          ssize_t  bytesWritten = Dwm::DescriptorIO::Write(fd, _progName);
          if (Dwm::IOUtils::StreamedLength(_progName) == bytesWritten) {
            rc = fixedSize + bytesWritten;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t RPCProgramId::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        if (Dwm::FileIO::Write(f, _progNum)) {
          if (Dwm::FileIO::Write(f, _protocol)) {
            if (Dwm::FileIO::Write(f, _version)) {
              rc = Dwm::FileIO::Write(f, _progName);
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint64_t RPCProgramId::StreamedLength() const
    {
      return (sizeof(_progNum) + sizeof(_protocol) + sizeof(_version)
              + Dwm::IOUtils::StreamedLength(_progName));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    nlohmann::json RPCProgramId::ToJson() const
    {
      nlohmann::json  j;
      j["num"]  = _progNum;
      j["prot"] = _protocol;
      j["ver"]  = _version;
      j["name"] = _progName;
      return j;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCProgramId::FromJson(const nlohmann::json & j)
    {
      bool  rc = false;
      if (j.is_object()) {
        auto  progNum = j.find("num");
        if ((progNum != j.end()) && progNum->is_number()) {
          _progNum = progNum->get<decltype(_progNum)>();
          auto  protocol = j.find("prot");
          if ((protocol != j.end()) && protocol->is_number()) {
            _protocol = protocol->get<decltype(_protocol)>();
            auto  version = j.find("ver");
            if ((version != j.end()) && version->is_number()) {
              _version = version->get<decltype(_version)>();
              auto  progName = j.find("name");
              if ((progName != j.end()) && progName->is_string()) {
                _progName = progName->get<decltype(_progName)>();
                rc = true;
              }
            }
          }
        }
      }
      return rc;
    }


  }  // namespace Mcrover
  
}  // namespace Dwm
