//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCUtils.cc 11977 $
// @(#) $Id: DwmMcroverRPCUtils.cc 11977 2022-04-24 18:42:11Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverRPCUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::RPCUtils class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <algorithm>
#include <thread>
#include <vector>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverRPCUtils.hh"
#include "DwmMcroverUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverRPCUtils.cc 11977 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCUtils::Test(map<RPCTarget,set<RPCProgramId>> & targets)
    {
      vector<thread>  threads;
      for (auto & t : targets) {
        threads.push_back(thread(&RPCTarget::Test, t.first,
                                 std::ref(t.second)));
      }
      for (auto & thr : threads) {
        if (thr.joinable()) {
          thr.join();
        }
        else {
          Syslog(LOG_ERR, "Thread not joinable!");
        }
      }
      return std::all_of(targets.begin(), targets.end(),
                         [&] (const pair<const RPCTarget,set<RPCProgramId>> & targ)
                         { return targ.second.empty(); });
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCUtils::Test(map<RPC6Target,set<RPCProgramId>> & targets)
    {
      vector<thread>  threads;
      for (auto & t : targets) {
        threads.push_back(thread(&RPC6Target::Test, t.first,
                                 std::ref(t.second)));
      }
      for (auto & thr : threads) {
        if (thr.joinable()) {
          thr.join();
        }
        else {
          Syslog(LOG_ERR, "Thread not joinable!");
        }
      }
      return std::all_of(targets.begin(), targets.end(),
                         [&] (const pair<const RPC6Target,set<RPCProgramId>> & targ)
                         { return targ.second.empty(); });
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCUtils::GetAlerts(const AlertOrigin & origin,
                             const vector<TargetHostConfig> & hosts,
                             AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      set<RPCTarget>  targets;
      for (const auto & host : hosts) {
        if (! host.ExpectedPrograms().empty()) {
          targets.insert(RPCTarget(host));
        }
      }
      if (! targets.empty()) {
        GetAlerts(origin, targets, alerts, clearAtEntry);
      }

      set<RPC6Target>  targets6;
      for (const auto & host : hosts) {
        if (! host.Expected6Programs().empty()) {
          targets6.insert(RPC6Target(host));
        }
      }
      if (! targets.empty()) {
        GetAlerts(origin, targets6, alerts, clearAtEntry);
      }
      
      return (! alerts.Empty());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCUtils::GetAlerts(const AlertOrigin & origin,
                             const set<RPCTarget> & targets,
                             AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      
      map<RPCTarget,set<RPCProgramId>>  testResults;
      for (const auto & target : targets) {
        testResults[target] = set<RPCProgramId>();
      }
      if (! Test(testResults)) {
        time_t  t = time((time_t *)0);
        char    thishost[256] = { '\0' };
        gethostname(thishost, 255);
        for (const auto & result : testResults) {
          for (const auto & prog : result.second) {
            RPCAlert  alert(result.first.Name(),
                            result.first.Address(), prog);
            alerts.Add(origin, alert, t);
          }
        }
      }
      return (! alerts.Empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RPCUtils::GetAlerts(const AlertOrigin & origin,
                             const set<RPC6Target> & targets,
                             AlertBowl & alerts, bool clearAtEntry)
    {
      if (clearAtEntry) {
        alerts.Clear();
      }
      
      map<RPC6Target,set<RPCProgramId>>  testResults;
      for (const auto & target : targets) {
        testResults[target] = set<RPCProgramId>();
      }
      if (! Test(testResults)) {
        time_t  t = time((time_t *)0);
        char    thishost[256] = { '\0' };
        gethostname(thishost, 255);
        for (const auto & result : testResults) {
          for (const auto & prog : result.second) {
            RPC6Alert  alert(result.first.Name(),
                             result.first.Address(), prog);
            alerts.Add(origin, alert, t);
          }
        }
      }
      return (! alerts.Empty());
    }
    

  }  // namespace Mcrover

}  // namespace Dwm
