//===========================================================================
// @(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverUPSTarget.cc 12374 $
// @(#) $Id: DwmMcroverUPSTarget.cc 12374 2024-05-31 03:40:15Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcroverUPSTarget.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Mcrover::UPSTarget class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <time.h>
  #include <stdint.h>
  #include <upsclient.h>
}

#include <algorithm>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcroverUPSTarget.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcrover/tags/mcrover-0.1.13/classes/src/DwmMcroverUPSTarget.cc 12374 $");

namespace Dwm {

  namespace Mcrover {

    using namespace std;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    UPSTarget::UPSTarget(const TargetHostConfig & hostConfig)
        : _name(hostConfig.Name()), _addr(hostConfig.Address()), _upses()
    {
      if (! hostConfig.UPS().empty()) {
        for (const auto & ups : hostConfig.UPS()) {
          _upses.insert(ups);
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string & UPSTarget::Name() const
    {
      return _name;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Ipv4Address & UPSTarget::Address() const
    {
      return _addr;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool UPSTarget::Test(map<string,UPSStatus> & result) const
    {
      bool  rc = false;
      for (const auto & ups : _upses) {
        result[ups] = UPSStatus::e_upsUnknown;
      }
      struct timeval  timeOut = { 2, 0 };
      UPSCONN_t       upsConn;
      if (upscli_tryconnect(&upsConn, _name.c_str(), 3493, 0, &timeOut) == 0) {
        for (const auto & ups : _upses) {
          const char *query[3] = { "VAR", ups.c_str(), "ups.status" };
#if (! defined(__APPLE__))
          size_t  numAnswers = 0;
#else
          //  MacPorts still has older nut-2.7.4
          unsigned int  numAnswers = 0;
#endif
          char  **answer;
          if (upscli_get(&upsConn, 3, query, &numAnswers, &answer) == 0) {
            if (numAnswers > 3) {
              result[ups] = UPSStatus(answer[3]);
            }
            else {
              result[ups] = UPSStatus(UPSStatus::e_upsUnknown);
            }
          }
          else {
            result[ups] = UPSStatus(UPSStatus::e_upsUnknown);
          }
        }
        if (upscli_disconnect(&upsConn) != 0) {
          Syslog(LOG_ERR, "upscli_disconnect() failed for %s", _name.c_str());
        }
      }
      return all_of(result.begin(), result.end(),
                    [&] (const pair<const string,UPSStatus> & r)
                    { return (r.second.Status()
                              == UPSStatus::e_upsOnLinePower); });
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool UPSTarget::operator < (const UPSTarget & target) const
    {
      bool  rc = false;
      if (_addr < target._addr) {
        rc = true;
      }
      else if (_addr == target._addr) {
        if (_name < target._name) {
          rc = true;
        }
        else if (_name == target._name) {
          if (_upses < target._upses) {
            rc = true;
          }
        }
      }
      return rc;
    }
    
          
    
  }  // namespace Mcrover

}  // namespace Dwm
